/* ***************************************************** **
   ch09_splitting_delimited_text.sql
   
   Skrypt dla książki Praktyczna nauka SQL dla Oracle, Helion (2022),
   napisanej przez Kima Berga Hansena, https://www.kibeha.dk
   Używasz na własną odpowiedzialność.
   *****************************************************
   
   Rozdział 9.
   Podział ograniczonego tekstu
   
   Skrypt przeznaczony do wykonania w schemacie PRACTICAL
** ***************************************************** */

/* -----------------------------------------------------
   Konfiguracja formatowania sqlcl
   ----------------------------------------------------- */

set pagesize 80
set linesize 80
set sqlformat ansiconsole

alter session set nls_date_format = 'YYYY-MM-DD';

/* -----------------------------------------------------
   Przykładowe fragmenty kodu do rozdziału 9.
   ----------------------------------------------------- */

-- Listing 9.1. Rozdzielona przecinkami treść w tabeli customer_favorites

select customer_id, favorite_list
from customer_favorites
order by customer_id;

-- Listing 9.3. Używanie potokowanej funkcji tabeli podczas podziału ciągu tekstowego

select
   cf.customer_id
 , fl.column_value as product_id
from customer_favorites cf
   , table(
        favorite_list_to_coll_type(cf.favorite_list)
     ) fl
order by cf.customer_id, fl.column_value;

-- Wyświetlenie samych klientów

select
   cf.customer_id
 , fl.column_value as product_id
from customer_favorites cf
   , table(
        favorite_list_to_coll_type(cf.favorite_list)
     )(+) fl
order by cf.customer_id, fl.column_value;

-- Listing 9.4. Złączenie z tabelą products wyniku podziału ciągu tekstowego

select
   cf.customer_id  as c_id
 , c.name          as cust_name
 , fl.column_value as p_id
 , p.name          as prod_name
from customer_favorites cf
cross apply table(
   favorite_list_to_coll_type(cf.favorite_list)
) fl
join customers c
   on c.id = cf.customer_id
join products p
   on p.id = fl.column_value
order by cf.customer_id, fl.column_value;

-- Wyświetlenie samych klientów - styl ANSI

select
   cf.customer_id  as c_id
 , c.name          as cust_name
 , fl.column_value as p_id
 , p.name          as prod_name
from customer_favorites cf
outer apply table(
   favorite_list_to_coll_type(cf.favorite_list)
) fl
join customers c
   on c.id = cf.customer_id
left outer join products p
   on p.id = fl.column_value
order by cf.customer_id, fl.column_value;

-- Listing 9.5. Podział ciągu tekstowego za pomocą funkcji apex_string.split()

select
   cf.customer_id  as c_id
 , c.name          as cust_name
 , to_number(fl.column_value) as p_id
 , p.name          as prod_name
from customer_favorites cf
cross apply table(
   apex_string.split(cf.favorite_list, ',')
) fl
join customers c
   on c.id = cf.customer_id
join products p
   on p.id = to_number(fl.column_value)
order by cf.customer_id, p_id;

-- Wyświetlenie samych klientów - styl ANSI

select
   cf.customer_id  as c_id
 , c.name          as cust_name
 , to_number(fl.column_value) as p_id
 , p.name          as prod_name
from customer_favorites cf
outer apply table(
   apex_string.split(cf.favorite_list, ',')
) fl
join customers c
   on c.id = cf.customer_id
left outer join products p
   on p.id = to_number(fl.column_value)
order by cf.customer_id, p_id;

--  Listing 9.6. Generowanie liczby rekordów odpowiadającej liczbie ograniczników

select
   favs.customer_id as c_id
 , c.name           as cust_name
 , favs.product_id  as p_id
 , p.name           as prod_name
from (
   select
      cf.customer_id
    , to_number(
         regexp_substr(cf.favorite_list, '[^,]+', 1, sub#)
      ) as product_id
   from customer_favorites cf
   cross join lateral(
      select level sub#
      from dual
      connect by level <= regexp_count(cf.favorite_list, ',') + 1
   ) fl
) favs
join customers c
   on c.id = favs.customer_id
join products p
   on p.id = favs.product_id
order by favs.customer_id, favs.product_id;

-- Obsługa sytuacji, w której ciąg tekstowy zawiera spacje

select
   favs.customer_id as c_id
 , c.name           as cust_name
 , favs.product_id  as p_id
 , p.name           as prod_name
from (
   select
      cf.customer_id
    , to_number(
         regexp_substr(
            cf.favorite_list
          , '(^|,)([^,]*)'
          , 1
          , sub#
          , null
          , 2
         )
      ) as product_id
   from customer_favorites cf
   cross join lateral(
      select level sub#
      from dual
      connect by level <= regexp_count(cf.favorite_list, ',') + 1
   ) fl
) favs
join customers c
   on c.id = favs.customer_id
join products p
   on p.id = favs.product_id
order by favs.customer_id, favs.product_id;

-- Listing 9.7. Traktowanie ciągu tekstowego jako tablicy JSON

select
   cf.customer_id  as c_id
 , c.name          as cust_name
 , fl.product_id   as p_id
 , p.name          as prod_name
from customer_favorites cf
outer apply json_table(
   '[' || cf.favorite_list || ']'
 , '$[*]'
   columns (
      product_id number path '$'
   )
) fl
join customers c
   on c.id = cf.customer_id
left outer join products p
   on p.id = fl.product_id
order by cf.customer_id, fl.product_id;

-- Listing 9.8. Rozdzielona przecinkami i dwukropkami treść tabeli customer_reviews zostanie wyświetlona po wykonaniu tego zapytania

select customer_id, review_list
from customer_reviews
order by customer_id;

--  Listing 9.10. Przykład użycia funkcji tabeli interfejsu ODCI do przetwarzania ograniczonych danych

select cr.customer_id, rl.product_id, rl.score
from customer_reviews cr
outer apply table (
   delimited_col_row.parser(
      cr.review_list
    , 'PRODUCT_ID:NUMBER,SCORE:VARCHAR2(1)'
    , ':'
    , ','
   )
) rl
order by cr.customer_id, rl.product_id;

-- Listing 9.11. Złączanie z nazwami rzeczywistych kolumn zamiast z ogólną pseudokolumną column_value

select
   cr.customer_id  as c_id
 , c.name          as cust_name
 , rl.product_id   as p_id
 , p.name          as prod_name
 , rl.score
from customer_reviews cr
cross apply table (
   delimited_col_row.parser(
      cr.review_list
    , 'PRODUCT_ID:NUMBER,SCORE:VARCHAR2(1)'
    , ':'
    , ','
   )
) rl
join customers c
   on c.id = cr.customer_id
join products p
   on p.id = rl.product_id
order by cr.customer_id, rl.product_id;

-- Listing 9.12. Pobieranie rekordów za pomocą wywołania apex_string.split() i kolumn za pomocą substr()

select
   cr.customer_id  as c_id
 , c.name          as cust_name
 , p.id            as p_id
 , p.name          as prod_name
 , substr(
      rl.column_value
    , instr(rl.column_value, ':') + 1
   ) as score
from customer_reviews cr
cross apply table(
   apex_string.split(cr.review_list, ',')
) rl
join customers c
   on c.id = cr.customer_id
join products p
   on p.id = to_number(
                substr(
                   rl.column_value
                 , 1
                 , instr(rl.column_value, ':') - 1
             ))
order by cr.customer_id, p_id;

-- Listing 9.13. Generowanie liczby rekordów odpowiadającej liczbie ograniczników

select
   revs.customer_id as c_id
 , c.name           as cust_name
 , revs.product_id  as p_id
 , p.name           as prod_name
 , revs.score
from (
   select
      cr.customer_id
    , to_number(
         regexp_substr(
            cr.review_list
          , '(^|,)([^:,]*)'
          , 1
          , sub#
          , null
          , 2
         )
      ) as product_id
    , regexp_substr(
         cr.review_list
       , '([^:,]*)(,|$)'
       , 1
       , sub#
       , null
       , 1
      ) as score
   from customer_reviews cr
   cross join lateral(
      select level sub#
      from dual
      connect by level <= regexp_count(cr.review_list, ',') + 1
   ) rl
) revs
join customers c
   on c.id = revs.customer_id
join products p
   on p.id = revs.product_id
order by revs.customer_id, revs.product_id;

-- Listing 9.14. Konwersja ograniczonego ciągu tekstowego na format JSON

select
   customer_id
 , '[["'
   || replace(
         replace(
            review_list
          , ','
          , '"],["'
         )
       , ':'
       , '","'
      )
   || '"]]'
   as json_list
from customer_reviews
order by customer_id;

-- Listing 9.15. Przetwarzanie danych JSON za pomocą funkcji json_table()

select
   cr.customer_id  as c_id
 , c.name          as cust_name
 , rl.product_id   as p_id
 , p.name          as prod_name
 , rl.score
from customer_reviews cr
cross apply json_table (
   '[["'
   || replace(
         replace(
            cr.review_list
          , ','
          , '"],["'
         )
       , ':'
       , '","'
      )
   || '"]]'
 , '$[*]'
   columns (
      product_id  number      path '$[0]'
    , score       varchar2(1) path '$[1]'
   )
) rl
join customers c
   on c.id = cr.customer_id
join products p
   on p.id = rl.product_id
order by cr.customer_id, rl.product_id;

-- Bonus: używanie obiektów JSON zamiast tablic

select
   cr.customer_id  as c_id
 , c.name          as cust_name
 , rl.product_id   as p_id
 , p.name          as prod_name
 , rl.score
from customer_reviews cr
cross apply json_table (
   nvl2(cr.review_list, '[{"p":', null)
   || replace(
         replace(
            replace(cr.review_list, ',', '|')
          , ':'
          , ',"r":"'
         )
       , '|'
       , '"},{"p":'
      )
   || nvl2(cr.review_list, '"}]', null)
 , '$[*]'
   columns (
      product_id  number      path '$.p'
    , score       varchar2(1) path '$.r'
   )
) rl
join customers c
   on c.id = cr.customer_id
join products p
   on p.id = rl.product_id
order by cr.customer_id, rl.product_id;

-- Bonus: konwersja tekstu ograniczonego na format XML

select
   customer_id
 , '<c>' || nvl2(review_list, '<o p="', null)
   || replace(
         replace(
            replace(review_list, ',', '|')
          , ':'
          , '" r="'
         )
       , '|'
       , '"/><o p="'
      )
   || nvl2(review_list, '"/>', null) || '</c>'
   as xml_list
from customer_reviews
order by customer_id;

-- Bonus: przetwarzanie danych XML za pomocą xmltable

select
   cr.customer_id  as c_id
 , c.name          as cust_name
 , rl.product_id   as p_id
 , p.name          as prod_name
 , rl.score
from customer_reviews cr
outer apply xmltable (
   '/c/o'
   passing xmltype(
      '<c>' || nvl2(cr.review_list, '<o p="', null)
      || replace(
            replace(
               replace(cr.review_list, ',', '|')
             , ':'
             , '" r="'
            )
          , '|'
          , '"/><o p="'
         )
      || nvl2(cr.review_list, '"/>', null) || '</c>'
   )
   columns
      product_id  number      path '@p'
    , score       varchar2(1) path '@r'
) rl
join customers c
   on c.id = cr.customer_id
left outer join products p
   on p.id = rl.product_id
order by cr.customer_id, rl.product_id;

/* ***************************************************** */
